<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);


include "./include/head.php";
include "sidebar.php";
include "./include/navbar.php";

if (!isset($_SESSION['userid'])) {
    header("Location: login.php");
    exit();
}

$success = $error = "";

// Fetch all available programs
$programs_q = mysqli_query($conn, "SELECT * FROM programs WHERE status='active' ORDER BY id ASC");
if (!$programs_q) {
    die("Database error (programs): " . mysqli_error($conn));
}
$programs = [];
while ($row = mysqli_fetch_assoc($programs_q)) {
    $programs[$row['id']] = $row;
}

// Fetch available wallets
$wallets = [];
$wallet_query = mysqli_query($conn, "SELECT * FROM wallet");
if ($wallet_query) {
    while ($row = mysqli_fetch_assoc($wallet_query)) {
        $wallets[$row['wallet_type']] = $row['wallet_address'];
    }
}

// Handle subscription form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $userid = $_SESSION['userid'];
    $program_id = intval($_POST['program_id'] ?? 0);
    $amount = floatval($_POST['amount'] ?? 0);
    $wallet_type = mysqli_real_escape_string($conn, $_POST['wallet_type'] ?? '');
    $wallet_address = mysqli_real_escape_string($conn, $_POST['wallet_address'] ?? '');
    $transaction_id = mysqli_real_escape_string($conn, $_POST['transaction_id'] ?? '');

    // basic server-side checks
    if (!$program_id || !isset($programs[$program_id])) {
        $error = "❌ Please select a valid program.";
    } else {
        $program = $programs[$program_id];
        $min = floatval($program['min_amount']);
        $max = floatval($program['max_amount']);

        if ($amount <= 0) {
            $error = "❌ Enter a valid amount.";
        } elseif ($amount < $min || $amount > $max) {
            $error = "❌ Amount must be between $" . number_format($min, 2) . " and $" . number_format($max, 2) . ".";
        } elseif (empty($wallet_type) || empty($wallet_address) || empty($transaction_id)) {
            $error = "❌ Please select a wallet and provide the transaction ID.";
        } else {
            // insert subscription
            $stmt = $conn->prepare("INSERT INTO program_subscriptions (userid, program_id, amount, wallet_type, wallet_address_input, transaction_id, roi, duration_days, status, created_at)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, 'pending', NOW())");
            if ($stmt) {
                $roi = intval($program['roi']);
                $duration = intval($program['duration_days']);
                $stmt->bind_param("sidsssii", $userid, $program_id, $amount, $wallet_type, $wallet_address, $transaction_id, $roi, $duration);

                if ($stmt->execute()) {
                    $success = "✅ Subscription submitted. Your transaction will be verified shortly.";
                    // optionally reset posted values so form clears
                    $_POST = [];
                } else {
                    $error = "❌ Failed to submit. Please try again.";
                }
                $stmt->close();
            } else {
                $error = "❌ Database error: " . $conn->error;
            }
        }
    }
}
?>

<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
<style>
    body {
        background: #f4f6fa;
    }

    .card-program {
        background: #fff;
        border-radius: 12px;
        box-shadow: 0 6px 18px rgba(0,0,0,0.06);
    }

    .btn-gradient {
        background: linear-gradient(135deg, #667eea, #764ba2);
        color: white;
        font-weight: 600;
    }

    .btn-copy {
        background: linear-gradient(135deg, #667eea, #764ba2);
        color: #fff;
        border: none;
    }

    label { font-weight: 600; }
    .program-meta { color: #333; }
</style>

<div class="container py-5">
    <div class="row justify-content-center">
        <div class="col-lg-8">

            <div class="text-center mb-4">
                <h2 class="fw-bold">💼 Programs</h2>
                <p class="text-muted">Select a program from the dropdown, enter the amount, complete payment, then submit your transaction details.</p>
            </div>

            <?php if ($success): ?>
                <div class="alert alert-success"><?php echo $success; ?></div>
            <?php elseif ($error): ?>
                <div class="alert alert-danger"><?php echo $error; ?></div>
            <?php endif; ?>

            <div class="card card-program p-4">
                <form method="POST" id="programForm" class="row g-3">
                    <div class="col-md-6">
                        <label for="program_id" class="form-label">Select Program</label>
                        <select name="program_id" id="program_id" class="form-control" required onchange="onProgramChange()">
                            <option value="" disabled selected>Choose a program</option>
                            <?php foreach ($programs as $id => $p): ?>
                                <option value="<?php echo $id; ?>"
                                    data-min="<?php echo htmlspecialchars($p['min_amount']); ?>"
                                    data-max="<?php echo htmlspecialchars($p['max_amount']); ?>"
                                    data-roi="<?php echo htmlspecialchars($p['roi']); ?>"
                                    data-duration="<?php echo htmlspecialchars($p['duration_days']); ?>"
                                    <?php if (isset($_POST['program_id']) && intval($_POST['program_id']) === intval($id)) echo 'selected'; ?>>
                                    <?php echo htmlspecialchars($p['program_name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <div class="col-md-6">
                        <label class="form-label">Program Details</label>
                        <div class="border rounded p-2 program-meta" id="programDetails">
                            <div>Min: <span id="p_min">-</span></div>
                            <div>Max: <span id="p_max">-</span></div>
                            <div>ROI: <span id="p_roi">-</span>%</div>
                            <div>Duration: <span id="p_duration">-</span> days</div>
                        </div>
                    </div>

                    <div class="col-md-6">
                        <label for="amount" class="form-label">Amount (USD)</label>
                        <input type="number" step="0.01" min="1" name="amount" id="amount" class="form-control" placeholder="Enter amount" required value="<?php echo isset($_POST['amount']) ? htmlspecialchars($_POST['amount']) : ''; ?>">
                        <small class="text-muted" id="amountHint"></small>
                    </div>

                    <div class="col-md-6">
                        <label for="wallet_type" class="form-label">Wallet Type</label>
                        <select name="wallet_type" id="wallet_type" class="form-control" required onchange="updateWalletAddress()">
                            <option value="" disabled selected>Select wallet</option>
                            <?php foreach ($wallets as $type => $address): ?>
                                <option value="<?php echo htmlspecialchars($type); ?>" <?php if (isset($_POST['wallet_type']) && $_POST['wallet_type'] === $type) echo 'selected'; ?>>
                                    <?php echo htmlspecialchars($type); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <div class="col-12">
                        <label class="form-label">Wallet Address</label>
                        <div class="input-group">
                            <input type="text" name="wallet_address" id="wallet_address" class="form-control" readonly required value="<?php echo isset($_POST['wallet_address']) ? htmlspecialchars($_POST['wallet_address']) : ''; ?>">
                            <button type="button" class="btn btn-copy" onclick="copyWalletAddress()" title="Copy address">
                                <i class="fas fa-copy"></i>
                            </button>
                        </div>
                    </div>

                    <div class="col-12">
                        <label for="transaction_id" class="form-label">Transaction ID</label>
                        <input type="text" name="transaction_id" id="transaction_id" class="form-control" required placeholder="Paste your transaction ID" value="<?php echo isset($_POST['transaction_id']) ? htmlspecialchars($_POST['transaction_id']) : ''; ?>">
                    </div>

                    <div class="col-12 text-center mt-2">
                        <button type="submit" class="btn btn-gradient rounded-pill px-5">Submit Payment</button>
                    </div>
                </form>
            </div>

            <div class="text-center mt-3 small text-muted">
                <a href="dashboard.php" class="text-decoration-none">&larr; Back to dashboard</a>
            </div>
        </div>
    </div>
</div>

<script>
    // programs data from server for JS use
    const programs = {};
    <?php foreach ($programs as $id => $p): ?>
        programs[<?php echo $id; ?>] = {
            min: parseFloat("<?php echo $p['min_amount']; ?>"),
            max: parseFloat("<?php echo $p['max_amount']; ?>"),
            roi: parseInt("<?php echo $p['roi']; ?>", 10),
            duration: parseInt("<?php echo $p['duration_days']; ?>", 10),
            name: "<?php echo addslashes($p['program_name']); ?>"
        };
    <?php endforeach; ?>

    const wallets = <?php echo json_encode($wallets); ?>;

    function onProgramChange() {
        const sel = document.getElementById('program_id');
        const id = sel.value ? parseInt(sel.value, 10) : null;
        const p_min = document.getElementById('p_min');
        const p_max = document.getElementById('p_max');
        const p_roi = document.getElementById('p_roi');
        const p_duration = document.getElementById('p_duration');
        const amountInput = document.getElementById('amount');
        const amountHint = document.getElementById('amountHint');

        if (id && programs[id]) {
            p_min.textContent = programs[id].min.toFixed(2);
            p_max.textContent = programs[id].max.toFixed(2);
            p_roi.textContent = programs[id].roi;
            p_duration.textContent = programs[id].duration;
            amountInput.min = programs[id].min;
            amountInput.max = programs[id].max;
            amountHint.textContent = `Allowed: $${programs[id].min.toFixed(2)} — $${programs[id].max.toFixed(2)}. ROI: ${programs[id].roi}% for ${programs[id].duration} days.`;
            // if current value is empty, prefill with min
            if (!amountInput.value || parseFloat(amountInput.value) < programs[id].min) {
                amountInput.value = programs[id].min.toFixed(2);
            }
        } else {
            p_min.textContent = '-';
            p_max.textContent = '-';
            p_roi.textContent = '-';
            p_duration.textContent = '-';
            amountHint.textContent = '';
            amountInput.min = 1;
            amountInput.max = '';
        }
    }

    function updateWalletAddress() {
        const type = document.getElementById("wallet_type").value;
        document.getElementById("wallet_address").value = wallets[type] || '';
    }

    function copyWalletAddress() {
        const addr = document.getElementById("wallet_address");
        if (!addr.value) {
            alert("No wallet address to copy.");
            return;
        }
        addr.select();
        addr.setSelectionRange(0, 99999);
        document.execCommand("copy");
        alert("✅ Wallet address copied to clipboard!");
    }

    // initialize if a program was preselected (e.g., after submit with error)
    window.addEventListener('DOMContentLoaded', () => {
        onProgramChange();
        // if wallet_type already selected from previous submit, update address
        const wt = document.getElementById('wallet_type');
        if (wt && wt.value) updateWalletAddress();
    });
</script>

<?php include "footer.php"; ?>
